<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\ServiceCategory;
use App\Models\ServiceArea;
use Illuminate\Http\Request;

class ServiceApiController extends Controller
{
    public function index()
    {
        $categories = ServiceCategory::with(['services.imageFile', 'detail.items.image', 'detail.heroImage', 'detail.section1Image1', 'detail.section1Image2', 'detail.section1Image3', 'detail.section2Image'])
            ->latest()
            ->get();
        return response()->json($categories);
    }

    /**
     * Build full image URL with encoded path (so filenames with spaces work).
     */
    private function imageUrl(?string $path): ?string
    {
        if (!$path || !is_string($path)) {
            return null;
        }
        $path = str_replace('\\', '/', trim($path, '/'));
        if ($path === '') {
            return null;
        }
        $encodedPath = implode('/', array_map('rawurlencode', explode('/', $path)));
        $base = rtrim(config('app.url'), '/');
        return $base ? $base . '/' . $encodedPath : null;
    }

    public function getCategoriesSimple()
    {
        $categories = ServiceCategory::with('image')
            ->select('id', 'name', 'slug', 'icon', 'description', 'image_id')
            ->latest()
            ->get();

        $categories->transform(function ($category) {
            $url = $this->imageUrl($category->image?->path ?? null);
            $category->setAttribute('image_url', $url);
            $category->makeVisible(['image_url']);
            return $category;
        });

        return response()->json([
            'status' => 'success',
            'data' => $categories
        ]);
    }

    public function getAreas()
    {
        $areas = ServiceArea::with('details')->latest()->get();
        return response()->json($areas);
    }

    public function getTopServices()
    {
        $topCategories = ServiceCategory::with(['services.imageFile', 'detail.items.image', 'detail.heroImage', 'detail.section1Image1', 'detail.section1Image2', 'detail.section1Image3', 'detail.section2Image'])
            ->where('is_top', true)
            ->latest()
            ->get();
        return response()->json($topCategories);
    }

    public function getService($slug)
    {
        $service = \App\Models\Service::with(['imageFile', 'category.detail.items.image', 'category.detail.heroImage', 'category.detail.section1Image1', 'category.detail.section1Image2', 'category.detail.section1Image3', 'category.detail.section2Image'])
            ->where('slug', $slug)
            ->firstOrFail();

        $service->service_areas = ServiceArea::with('details')->get();
        return response()->json($service);
    }

    public function getCategory($slug)
    {
        $category = ServiceCategory::with(['services.imageFile', 'detail.items.image', 'detail.heroImage', 'detail.section1Image1', 'detail.section1Image2', 'detail.section1Image3', 'detail.section2Image'])
            ->where('slug', $slug)
            ->firstOrFail();
        return response()->json($category);
    }

    public function getArea($slug)
    {
        $area = ServiceArea::with('details')->where('slug', $slug)->firstOrFail();
        return response()->json($area);
    }
}
