<?php

namespace App\Http\Controllers\Admin;

use App\Models\Category;
use App\Models\Product;
use App\Models\Tag;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Str;

class ProductController extends Controller
{
    public function index()
    {
        $products = Product::with('category', 'thumbnail')->latest()->paginate(10);
        return view('admin.products.index', compact('products'));
    }

    public function create()
    {
        $categories = Category::all();
        $tags = Tag::all();
        return view('admin.products.create', compact('categories', 'tags'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'category_id' => 'required|exists:categories,id',
            'title' => 'required|string|max:255',
            'slug' => 'nullable|string|unique:products,slug',
            'thumbnail_id' => 'nullable|exists:images,id',
            'price' => 'required|numeric',
            'discount_price' => 'nullable|numeric',
            'short_description' => 'nullable|string',
            'long_description' => 'nullable|string',
            'stock_quantity' => 'nullable|integer|min:0',
            'status' => 'required|string|in:in_stock,out_of_stock',
            'gallery_ids' => 'nullable|string', // JSON array of image IDs
            'tag_ids' => 'nullable|array',
            'tag_ids.*' => 'exists:tags,id',
        ]);

        if (empty($data['slug'])) {
            $data['slug'] = Str::slug($data['title']);
        }

        $product = Product::create($data);

        if ($request->gallery_ids) {
            $imageIds = json_decode($request->gallery_ids);
            $product->images()->sync($imageIds);
        }

        if ($request->tag_ids) {
            $product->tags()->sync($request->tag_ids);
        }

        return redirect()->route('admin.products.index')->with('success', 'Product created successfully');
    }

    public function edit(Product $product)
    {
        $categories = Category::all();
        $tags = Tag::all();
        $product->load('images', 'tags');
        return view('admin.products.edit', compact('product', 'categories', 'tags'));
    }

    public function update(Request $request, Product $product)
    {
        $data = $request->validate([
            'category_id' => 'required|exists:categories,id',
            'title' => 'required|string|max:255',
            'slug' => 'nullable|string|unique:products,slug,' . $product->id,
            'thumbnail_id' => 'nullable|exists:images,id',
            'price' => 'required|numeric',
            'discount_price' => 'nullable|numeric',
            'short_description' => 'nullable|string',
            'long_description' => 'nullable|string',
            'stock_quantity' => 'nullable|integer|min:0',
            'status' => 'required|string|in:in_stock,out_of_stock',
            'gallery_ids' => 'nullable|string',
            'tag_ids' => 'nullable|array',
            'tag_ids.*' => 'exists:tags,id',
        ]);

        if (empty($data['slug'])) {
            $data['slug'] = Str::slug($data['title']);
        }

        $product->update($data);

        if ($request->gallery_ids) {
            $imageIds = json_decode($request->gallery_ids);
            $product->images()->sync($imageIds);
        }

        if ($request->tag_ids) {
            $product->tags()->sync($request->tag_ids);
        }

        return redirect()->route('admin.products.index')->with('success', 'Product updated successfully');
    }

    public function destroy(Product $product)
    {
        $product->delete();
        return back()->with('success', 'Product deleted successfully');
    }
}
